/* SERISR.C: Serial interrupt service routines      */
/* By: Martin Grotegut 1989-1990                    */
/* To be compiled using older Microsoft-C compilers */

/* Includes */
#include <stdio.h>
#include <stdlib.h>
#include <dos.h>
#include <conio.h>
#include "portab.h"

/* Function prototypes */
void interrupt far dummy_iret(void);
void init_rs232(void);
void init_base_addr(void);
void reset_rs232(void);

/* 8250 register defines (add these offsets to base addr)		*/
#define RBR 0		   	/* Receive Buffer Register READ ONLY	*/
#define THR 0		   	/* Transmitter Hold Register WRITE ONLY */
#define IER 1		   	/* Interrupt Enable Register		*/
#define IIR 2		   	/* Interrupt ID Register		*/
#define LCR 3		   	/* Line Control Register		*/
#define MCR 4		   	/* Modem Control Register		*/
#define LSR 5		   	/* Line Status Register			*/
#define MSR 6		   	/* Modem Status Register		*/
#define DLAR 0		   	/* Divisor Low (Bits 0-7)		*/
#define DHAR 1		   	/* Divisor High (Bits 8-15)		*/

/* Pragmas */
#pragma intrinsic(inp, outp, _enable, _disable)

extern void (interrupt far *old_intC)(void);	/* COM1: interrupt handler */
extern void (interrupt far *old_int1B)(void);	/* CTRL-Break interrupt handler */
extern UWORD (interrupt far *old_int14)(void);	/* BIOS Serial I/O Int */

/* Local globals */
static const UWORD baud_rate_divisor[8]={1047, 768, 384, 192, 96, 48, 24, 12};
static UWORD rs232_base[4];		/* 8250 Base Addresses array */
static volatile UCHAR com_buff_head;
static volatile UCHAR com_buff_tail;
static volatile UWORD com_buff[256];		/* 256 chars + status bytes */
static union REGS regs;


void interrupt far new_intC(unsigned es, unsigned ds, unsigned di,
			    unsigned si, unsigned bp, unsigned sp,
			    unsigned bx, unsigned dx, unsigned cx,
			    unsigned ax, unsigned ip, unsigned cs,
			    unsigned flags)
{
	_disable();
	if( ( inp( 0x3F8 + IIR ) & 0x06) == 4 ) /* Test for Data Read Intr */
	{
	 /* Read char from RCV buffer reg. and setup status in AH */
	 com_buff[com_buff_head++] = ( inp( 0x3F8 + RBR ) & 0xFF ) |
				     ( ( inp( 0x3F8 + LSR ) & 0x1E ) << 8 );
	 outp(0x20, 0x20);		/* Flag 8259 End-of-intr */
	}
	_enable();
}


UWORD interrupt far new_int14(unsigned es, unsigned ds, unsigned di,
				 unsigned si, unsigned bp, unsigned sp,
				 unsigned bx, unsigned dx, unsigned cx,
				 unsigned ax, unsigned ip, unsigned cs,
				 unsigned flags)
{
UWORD	tmp_base;

	_disable();
	dx &= 0x03;			/* 4 COM ports allowed */
	if( tmp_base = rs232_base[dx] ) /* If 8250 exists */
	{
	 switch( ax >> 8 )
	 {
	  case 0x00:			/* Set comm. params */
	 	 outp(tmp_base+LCR, 0x80); /* Set DLAB bit */
		 outp(tmp_base+DLAR, baud_rate_divisor[ax>>5]&0xFF);
	 	 outp(tmp_base+DHAR, baud_rate_divisor[ax>>5]>>8);
		 outp(tmp_base+LCR, ax & 0x1F);	/* Set remaining bits */
		 /* Fall through to ... */
	  case 0x03:			/* Status request */
	  	 /* Get status */
		 ax = (inp(tmp_base+LSR)<<8) | (inp(tmp_base+MSR)&0xFF);
		 if( com_buff_head == com_buff_tail )	/* Chars in buffer ? */
		  ax &= 0xFEFF;		/* Say buffer empty */
		 else
		  ax |= 0x0100;		/* Fake data ready */
		 break;

	  case 0x01:			/* Send char in AL */
		 ax &= 0xFF;
		 while( !( inp( tmp_base + LSR ) & 0x20 ) )
		  ;			/* Wait for THRE empty */
		 outp(tmp_base+THR, ax); /* Send char */
		 ax |= inp( tmp_base + LSR ) << 8; /* 'Or' in status */
		 break;

	  case 0x02:			/* Receive char */
		 if( com_buff_head != com_buff_tail) /* Buffer not empty */
		  ax = com_buff[com_buff_tail++]; /* Get char & status */
		 else
		  ax = ( inp( tmp_base + LSR ) << 8 ) | 0x8000; /* Time out */
		 break;

	 }
	}
	_enable();
	return( ax );			/* Write ax into reg. AX */
}



void interrupt far dummy_iret( void )
{
	return;
}


/* Enable the 8250 UART to generate interrupts */
void init_rs232(void)
{
	_disable();
	com_buff_head = com_buff_tail = 0;
	init_base_addr();		/* Get 8250 base addresses */

	/* Get old routine vectors */
	old_intC  = _dos_getvect( 0x0C ); /* Hex C; COM1: */
	old_int14 = _dos_getvect( 0x14 );

	/* Point to new routines */
	_dos_setvect(0x0C, new_intC);
	_dos_setvect(0x14, new_int14);

	outp(0x3F8+LCR,inp(0x3F8+LCR)&0x7F); /* Reset Div. Latch Access Bit */
	outp(0x3F8+IER,inp(0x3F8+IER)|0x01); /* Enable Rec. data int. 	    */
	outp(0x3F8+MCR,inp(0x3F8+MCR)|0x0B); /* Set DTR, RTS, Intr enable   */
	outp(0x21, inp(0x21) & 0xEF);	/* Enable 8259 COM1 interrupts      */
	inp( 0x3F8 );			/* RCV twice to clear old char and  */
	inp( 0x3F8 );			/* interrupt that may be pending    */
	_enable();
}


/* Fills base addresses array with proper 8250 base adresses */
/* If a base address is zero, an 8250 SIO chip was not found */
void init_base_addr(void)
{
UWORD far *ptr;
UWORD i;

	ptr = (UWORD far *) 0x00000400; /* DOS copy of 8250 base addr */
	for(i = 0; i < 4; i++)
	 rs232_base[i] = *ptr++;
}


void reset_rs232(void)
{
	_disable();
	outp(0x3F8+IER, 0);		/* Disable all 8250 interrupts	*/
	outp(0x3F8+MCR, 0);		/* Reset DTR, RTS, Intr enable	*/
	outp(0x21, inp(0x21) | 0x10);	/* Disable 8259 COM1 interrupts */
	/* Restore interrupt vectors */
	_dos_setvect(0x0C, old_intC);
	_dos_setvect(0x14, old_int14);
	inp( 0x3F8 );			/* RCV twice to clear old char and  */
	inp( 0x3F8 );			/* interrupt that may be pending    */
	_enable();
}

